%{ open Ast %}

/* Tokens */
%token EOL EOF
%token PLUS MINUS MULTIPLY DIVIDE MODULUS
%token LT LTE GT GTE EQ NEQ TRUE FALSE
%token AND OR NOT
%token LPAREN RPAREN LBRACE RBRACE LSQUARE RSQUARE
%token ASSIGN LAMBDA END IF ELIF ELSE
%token CARET CONS APPEND
%token COLON COMMA
%token NONE
%token <float> NUM_LIT
%token <string> STR_LIT
%token <string> ID

%right ASSIGN
%right CONS
%left APPEND
%left CARET
%left OR
%left AND
%left NOT
%left LTE GTE LT GT EQ NEQ
%left PLUS MINUS
%left MULTIPLY DIVIDE MODULUS
%left NEG

%start program
%type <Ast.program> program

%%

program:
  expr_list EOF   { $1 }

delimited_expr:
    EOL           { NoExpr }
  | expr EOL      { $1 }

expr_list:
  /* nothing */              { [] }
  | delimited_expr expr_list { if $1 = NoExpr then $2 else $1 :: $2 }

fun_literal:
  LAMBDA LPAREN formals_opt RPAREN COLON EOL expr_list END  { FunLit($3, Block($7)) }

formals_opt:
  /* nothing */  { [] }
  | formal_list  { List.rev $1 }

formal_list:
  ID                      { [$1] }
  | formal_list COMMA ID  { $3 :: $1 }

if_expr:
  IF expr COLON EOL expr_list elif_list ELSE COLON EOL expr_list END  { If(($2, Block($5)) :: $6, Block($10)) }

elif_list:
  /* nothing */     { [] }
  | elif elif_list  { $1 :: $2 }

elif:
  ELIF expr COLON EOL expr_list  { ($2, Block($5)) }

actuals_opt:
  /* nothing */  { [] }
  | actual_list  { List.rev $1 }

actual_list:
    expr                    { [$1] }
  | actual_list COMMA expr  { $3 :: $1 }

kv_pairs:
  /* nothing */   { [] }
  | kv_pair_list  { List.rev $1 }

kv_pair_list:
    expr COLON expr                     { [($1, $3)] }
  | kv_pair_list COMMA expr COLON expr  { ($3, $5) :: $1 }

literals:
    NUM_LIT   { NumLit($1) }
  | STR_LIT   { StringLit($1) }
  | TRUE      { BoolLit(true) }
  | FALSE     { BoolLit(false) }
  | ID        { Val($1) }
  | NONE      { VoidLit }
  | LSQUARE actuals_opt RSQUARE  { ListLit($2) }
  | LBRACE kv_pairs RBRACE       { DictLit($2) }
  | fun_literal   { $1 }

expr:
  literals              { $1 }
  | if_expr             { $1 }
  | expr PLUS expr      { Binop($1, Add, $3) }
  | expr MINUS expr     { Binop($1, Sub, $3) }
  | expr MULTIPLY expr  { Binop($1, Mult, $3) }
  | expr DIVIDE expr    { Binop($1, Div, $3) }
  | expr MODULUS expr   { Binop($1, Mod, $3) }
  | expr EQ expr        { Binop($1, Eq, $3) }
  | expr NEQ expr       { Binop($1, Neq, $3) }
  | expr GT expr        { Binop($1, Gt, $3) }
  | expr GTE expr       { Binop($1, Gte, $3) }
  | expr LT expr        { Binop($1, Lt, $3) }
  | expr LTE expr       { Binop($1, Lte, $3) }
  | expr AND expr       { Binop($1, And, $3) }
  | expr OR expr        { Binop($1, Or, $3) }
  | expr CONS expr      { Binop($1, Cons, $3) }
  | expr APPEND expr    { Binop($1, Append, $3) }
  | expr CARET expr     { Binop($1, Caret, $3) }
  | MINUS expr %prec NEG  { Unop(Neg, $2) }
  | NOT expr            { Unop(Not, $2) }
  | ID ASSIGN expr      { Assign($1, $3) }
  | LPAREN expr RPAREN  { $2 }
  | ID LPAREN actuals_opt RPAREN  { Call(Val($1), $3) }
  | ID LSQUARE expr RSQUARE       { Element($1, $3) }
